function deepEqual(obj1, obj2) {
    if (obj1 === obj2) return true;
    if (typeof obj1 !== 'object' || typeof obj2 !== 'object' || obj1 === null || obj2 === null) return false;

    const keys1 = Object.keys(obj1);
    const keys2 = Object.keys(obj2);

    if (keys1.length !== keys2.length) return false;

    for (let key of keys1) {
        if (!keys2.includes(key) || !deepEqual(obj1[key], obj2[key])) return false;
    }

    return true;
}

const dragDropUI = {
    grid: document.getElementById('grid'),
    itemsContainer: document.getElementById('items'),
    contextMenu: document.getElementById('context-menu'),
    items: new Proxy([], {
        set(target, property, value) {
            const originalItem = target[property];
            if (property !== 'length' && !deepEqual(originalItem, value)) {
                dragDropUI.saveSchemeToFile();
            }
            target[property] = value;
            return true;
        },
        get(target, property) {
            const value = target[property];
            if (typeof value === 'object' && value !== null) {
                return new Proxy(value, {
                    set(obj, prop, val) {
                        if (!deepEqual(obj[prop], val)) {
                            dragDropUI.saveSchemeToFile();
                        }
                        obj[prop] = val;
                        return true;
                    }
                });
            }
            return value;
        }
    }),
    options: {
        defaultActiveColor: 'red',
        defaultPassiveColor: 'transparent',
        strings: {
            selectActiveColor: 'Please active state select color:',
            selectPassiveColor: 'Please passive state select color:',
            selectLogicColor: 'Please logic state select color:',
            selectId: 'Please select ID:',
            save: 'Save',
            edit: 'Edit',
            add: 'Add',
            item: 'item',
            text: 'text',
            apply: 'apply',
            view: 'view',
            selectPredefinedGrid: 'Select a Predefined Grid'
        }
    },
    fromHTML: function (html, trim = true) {
        html = trim ? html.trim() : html;
        if (!html) return null;
        const template = document.createElement('template');
        template.innerHTML = html;
        const result = template.content.children;
        if (result.length === 1) return result[0];
        return result;
    },
    safeStringify: function (obj) {
        return JSON.stringify(obj, function (key, value) {
            return typeof value === 'string'
                ? encodeURIComponent(value)
                : value;
        });
    },
    safeParse: function (jsonString) {
        try {
            const parsed = JSON.parse(jsonString);

            function decodeStrings(obj) {
                if (typeof obj === 'string') {
                    return decodeURIComponent(obj);
                } else if (Array.isArray(obj)) {
                    return obj.map(decodeStrings);
                } else if (obj && typeof obj === 'object') {
                    return Object.keys(obj).reduce((acc, key) => {
                        acc[key] = decodeStrings(obj[key]);
                        return acc;
                    }, {});
                }
                return obj;
            }

            return decodeStrings(parsed);
        } catch (e) {
            console.error('Error parsing JSON:', e);
            return null;
        }
    },
    clearScheme: function () {
        const currentItems = document.querySelectorAll('.cell .item');
        currentItems.forEach(item => item.remove());
        this.items.length = 0;
    },
    exportScheme: async function () {
        try {
            await java.download('scheme_export' + Date.now() + '.json');
        } catch (e) {
            Toastify({
                text: e.message, style: {
                    background: '#fa4a4a'
                }
            }).showToast();
            console.error('Error calling java.download:', e);
        }
        return null;
    },
    handleImportScheme: function (event) {
        const that = this;
        const file = event.target.files[0];
        if (!file) return;

        const reader = new FileReader();
        reader.onload = function (event) {
            const fileContent = event.target.result;

            try {
                const result = JSON.parse(fileContent);

                if (result.items?.length) {
                    result.items.forEach(resItem => that.items.push(resItem))
                } else {
                    Toastify({
                        text: "Wrong file format", style: {
                            background: '#fa4a4a'
                        }
                    }).showToast();
                    return;
                }

                that.deleteScheme()

                that.xCells = result.xCells || that.xCells;
                that.yCells = result.yCells || that.yCells;

                that.createScheme(that.xCells, that.yCells);
                that.applySchemeItems(result.items);
                Toastify({
                    text: 'Success', style: {
                        background: '#047c00'
                    }
                }).showToast();
            } catch (e) {
                Toastify({
                    text: e.message, style: {
                        background: '#fa4a4a'
                    }
                }).showToast();
                console.error('Error importing file:', e);
            }
            event.target.value = "";
        };
        reader.readAsText(file);
    },
    saveSchemeToFile: async function () {

        const schemeData = {
            items: this.items,
            xCells: this.xCells,
            yCells: this.yCells
        };
        const jsonData = JSON.stringify(schemeData);
        try {
            if (typeof java !== 'undefined') {
                await java.saveScheme(schemeData);
            }
        } catch (e) {
            Toastify({
                text: e.message, style: {
                    background: '#fa4a4a'
                }
            }).showToast();
            console.error('Error calling java.saveScheme:', e);
        }
    },
    applySchemeItems: function (itemsArray) {
        this.clearScheme();
        itemsArray.forEach((item) => {
            const itemDiv = this.createItemElement({
                item
            });
            this.placeItem(itemDiv, item.x, item.y);
            this.items.push(item);
        });
    },
    deleteScheme: function () {
        this.grid.innerHTML = '';
    },
    createScheme: function (xCells, yCells) {
        this.xCells = xCells;
        this.yCells = yCells;

        this.grid.style.gridTemplateColumns = `repeat(${xCells}, 50px)`;
        this.grid.style.gridTemplateRows = `repeat(${yCells}, 50px)`;

        for (let i = 0; i < xCells * yCells; i++) {
            const cell = document.createElement('div');
            cell.classList.add('cell');
            cell.dataset.index = i;
            cell.addEventListener('dragover', this.dragOver.bind(this));
            cell.addEventListener('dragleave', this.dragLeave.bind(this));
            cell.addEventListener('drop', this.drop.bind(this));
            this.grid.appendChild(cell);
        }
    },
    init: async function (xCells, yCells) {
        this.grid = document.getElementById('grid');
        this.xCells = xCells;
        this.yCells = yCells;

        try {
            if (window.initialScheme) {
                const schemeData = window.initialScheme;
                this.xCells = schemeData.xCells || xCells;
                this.yCells = schemeData.yCells || yCells;
                this.createScheme(schemeData.xCells || xCells, schemeData.yCells || yCells);
                this.applySchemeItems(schemeData.items);
            } else {
                this.createScheme(xCells, yCells);
            }
        } catch (e) {
            this.createScheme(xCells, yCells);
        }

        document.getElementById('importScheme').onclick = function () {
            this.value = null;
        };
    },
    createItemElement: function ({
                                     item,
                                     isDash = false
                                 }) {
        const {
            id,
            imageName,
            rotation,
            mirror,
            active,
            passiveColor: itemPassiveColor,
            activeColor: itemActiveColor,
            name
        } = item;

        const itemDiv = document.createElement('div');
        itemDiv.classList.add('item');
        if (active) {
            itemDiv.classList.add('active');
        }

        let activeColor = this.options.defaultActiveColor;
        let passiveColor = this.options.defaultPassiveColor;
        if (itemPassiveColor) activeColor = itemPassiveColor;
        if (itemActiveColor) activeColor = itemActiveColor;
        itemDiv.style.backgroundColor = active ? activeColor : passiveColor;

        if (imageName && images[imageName]) {
            itemDiv.style.backgroundImage = images[imageName];
        }
        itemDiv.setAttribute('draggable', true);
        itemDiv.id = id;
        itemDiv.dataset.category = item.active ? 'active' : item.category;
        itemDiv.dataset.state = 'undefined';
        itemDiv.dataset.imageName = item.imageName;
        itemDiv.style.transform = `rotate(${rotation || 0}deg) ${mirror ? 'scaleX(-1)' : ''}`;
        itemDiv.addEventListener('dragstart', this.dragStart.bind(this));
        itemDiv.addEventListener('dragend', this.dragEnd.bind(this));
        itemDiv.addEventListener('contextmenu', this.showContextMenu.bind(this));

        if (isDash && name) {
            itemDiv.title = name;
            itemDiv.dataset.title = name;
        }
        if (!isDash) itemDiv.addEventListener('click', () => this.toggleActiveState(itemDiv));

        if (item.imageName === 'text' && item.textData) {
            itemDiv.classList.add('text-item');
            itemDiv.draggable = true;
            const deltaConverter = new QuillDeltaToHtmlConverter(item.textData, {});
            itemDiv.innerHTML = deltaConverter.convert();
        }
        return itemDiv;
    },
    placeItem: function (item, x, y) {
        const cellIndex = y * this.xCells + x;
        this.grid.children[cellIndex].appendChild(item);
    },
    dragStart: function (e) {
        let elementsList = document.getElementById('modal-content-el');
        if (elementsList || (e.target.parentElement === this.modal.content)) {
            const clonedItem = e.target.cloneNode(true);

            clonedItem.classList.add('is-dragging');

            clonedItem.id = `cloned-${!clonedItem.id || clonedItem.id === 'undefined' ? Date.now() : clonedItem.id}`;
            clonedItem.style.position = 'absolute';
            clonedItem.style.pointerEvents = 'none';
            document.body.appendChild(clonedItem);

            const onDrag = (event) => {
                clonedItem.style.left = `${event.pageX - clonedItem.offsetWidth / 2}px`;
                clonedItem.style.top = `${event.pageY - clonedItem.offsetHeight / 2}px`;
            };
            document.addEventListener('dragover', onDrag);

            e.target.addEventListener('dragend', () => {
                document.removeEventListener('dragover', onDrag);
                clonedItem.classList.remove('is-dragging');
                clonedItem.style.position = 'static';
                clonedItem.style.pointerEvents = 'inherit';
                clonedItem.style.left = 0;
                clonedItem.style.top = 0;
                clonedItem.id = clonedItem.id.replace('cloned-', '');
            });

            clonedItem.addEventListener('dragstart', this.dragStart.bind(this));
            clonedItem.addEventListener('dragend', this.dragEnd.bind(this));
            clonedItem.addEventListener('contextmenu', this.showContextMenu.bind(this));
            clonedItem.addEventListener('click', () => this.toggleActiveState(clonedItem));
            this.draggedElement = clonedItem;
        } else {
            e.dataTransfer.setData('text/plain', e.target.id);
            this.draggedElement = e.target;
            this.draggedElement.originalParent = this.draggedElement.parentElement;
            setTimeout(() => {
                e.target.style.display = 'none';
            }, 0);
        }
    },
    dragEnd: function (e) {
        e.target.style.display = 'block';
        this.draggedElement = null;
    },
    dragOver: function (e) {
        e.preventDefault();

        const targetCell = e.target.classList.contains('cell') ? e.target : e.target.closest('.cell');

        if (targetCell) {
            if (targetCell.hasChildNodes()) {
                targetCell.classList.add('highlight-red');
            } else {
                targetCell.classList.add('highlight');
            }
        }
    },

    dragLeave: function (e) {
        const targetCell = e.target.classList.contains('cell') ? e.target : e.target.closest('.cell');

        if (targetCell.classList.contains('cell')) {
            targetCell.classList.remove('highlight');
            targetCell.classList.remove('highlight-red');
        }
    },
    drop: function (e) {
        e.preventDefault();
        const targetCell = e.target.classList.contains('cell') ? e.target : e.target.closest('.cell');

        if(targetCell){
            targetCell.classList.remove('highlight');
            targetCell.classList.remove('highlight-red');

            if (targetCell.hasChildNodes()) {
                // If the target cell is not empty, return the dragged item to its original cell
                if (this.draggedElement && this.draggedElement.originalParent) {
                    this.draggedElement.originalParent.appendChild(this.draggedElement);
                }
                return; // Exit without placing the item
            }

            targetCell.appendChild(this.draggedElement);
            this.draggedElement.id = this.draggedElement.id.replace('cloned-', '');
            const cellIndex = Array.from(this.grid.children).indexOf(e.target);
            const x = cellIndex % this.xCells;
            const y = Math.floor(cellIndex / this.xCells);
            const item = this.items.find((item) => item.id === this.draggedElement.id);
            if (item) {
                item.x = x;
                item.y = y;
            } else {
                const newItem = {
                    id: this.draggedElement.id,
                    x: x,
                    y: y,
                    rotation: 0,
                    mirror: false,
                    imageName: this.draggedElement.dataset.imageName || 'default',
                    active: false
                };
                this.items.push(newItem);
            }
            this.saveSchemeToFile();
        }
    },
    deleteItem: function (item) {
        if (item.parentElement && item.parentElement.classList.contains('cell')) {
            const index = this.items.findIndex((i) => i.id === item.id);
            if (index !== -1) {
                this.items.splice(index, 1);
            }
            item.parentElement.removeChild(item);
        }
    },
    findItemById(id) {
        return this.items.find(item => item.id === id);
    },
    showContextMenu: function (e) {
        e.preventDefault();
        if (this.isInCell(e.target)) {
            this.contextMenu.style.top = `${e.clientY}px`;
            this.contextMenu.style.left = `${e.clientX}px`;
            this.contextMenu.style.display = 'block';
            this.contextMenuTarget = e.target;

            document.addEventListener('click', this.hideContextMenu.bind(this));

            document.querySelectorAll('.context-menu__item').forEach(element => {
                element.addEventListener('click', this.contextAction.bind(this))
            })
        }
    },
    hideContextMenu: function () {
        this.contextMenu.style.display = 'none';
    },
    isInCell: function (item) {
        return item.parentElement && (item.closest('.cell'));
    },
    contextAction: function (event) {
        //  'rotate' | 'edit' | 'mirror' | 'delete'
        const actionType = event.target.dataset.action;
        const item = this.contextMenuTarget.closest('.item') || this.contextMenuTarget.closest('.text-item');
        const foundItem = this.findItemById(item.id)
        const currentTransform = item.style.transform;

        switch (actionType) {
            case 'rotate':
                const rotation = currentTransform.includes('rotate') ? parseInt(currentTransform.match(/rotate\((\d+)deg\)/)[1]) : 0;
                const newRotation = (rotation + 90) % 360;
                item.style.transform = `rotate(${newRotation}deg)`;
                if (foundItem) foundItem.rotation = newRotation;
                break;
            case 'mirror':
                const isMirrored = currentTransform.includes('scaleX(-1)');
                const newTransform = isMirrored ? currentTransform.replace('scaleX(-1)', '') : currentTransform + ' scaleX(-1)';
                item.style.transform = newTransform.trim();
                foundItem.mirror = isMirrored;
                break;
            case 'edit':
                if (item.classList.contains('text-item')) {
                    this.showTextModal(item);
                } else {
                    this.showDynamicItemModal(item);
                }
                break;
            case 'delete':
                this.deleteItem(item);
                break;
            default:
                break;
        }
        this.hideContextMenu();

    },
    toggleActiveState: function (item) {
        const active = !item.classList.contains('active');
        item.classList.toggle('active', active);

        const foundItem = this.findItemById(item.id)
        if (foundItem) foundItem.active = active;

        let activeColor = this.options.defaultActiveColor;
        let passiveColor = this.options.defaultPassiveColor;
        if (foundItem) {
            if (foundItem.activeColor) activeColor = foundItem.activeColor;
            if (foundItem.passiveColor) passiveColor = foundItem.passiveColor;
            foundItem.active = active;
        }

        const category = item.getAttribute("data-category");
        if (category === 'active') {
            this.changeActiveEl(item)
        }

        item.style.backgroundColor = active ? activeColor : passiveColor;
    },
    modal: {
        isVisible: false,
        title: document.getElementById('modal-title'),
        content: document.getElementById('modal-content'),
        wrapper: document.getElementById('modal'),
        hide: function () {
            this.title.innerHTML = '';
            this.content.innerHTML = '';
            this.isVisible = false;
            this.wrapper.style.display = 'none';
        },
        show: function (content, title) {
            this.title.innerHTML = title;
            if (Array.isArray(content)) {
                content.forEach(contentItem => this.content.appendChild(contentItem))
            } else {
                this.content.appendChild(content)
            }
            this.isVisible = true;
            this.wrapper.style.display = 'block';
        },
    },
    showDashboardModal: function () {
        this.modal.hide();
        const dashItems = dashboardItems.map((item) => this.createItemElement({
            item,
            isDash: true,
        }));

        const categories = [...new Set(dashboardItems.map(x => x.category))];
        let mainList = document.createElement('ul');
        mainList.id = 'modal-content-el';
        categories.forEach(cat => {
            let staticList = document.createElement('li');
            let spanStaticEl = document.createElement('span');
            spanStaticEl.classList.add('caret');
            spanStaticEl.innerHTML = cat + ' elements';
            staticList.appendChild(spanStaticEl);

            let ul = document.createElement('ul');
            ul.classList.add('nested');

            dashItems.forEach(item => {
                const category = item.getAttribute("data-category");
                if (category === cat) {
                    let li = document.createElement('li');
                    li.appendChild(item);
                    ul.appendChild(li)
                }
            });

            staticList.appendChild(ul);
            mainList.appendChild(staticList);
        });

        this.modal.show(mainList, 'Items')
        this.addEventListenerForElementsList();
    },
    showDynamicItemModal: function (item) {
        this.modal.hide();

        const itemId = item.id.replace('cloned-', '').split('-')[0];
        const foundItem = this.items.find(i => i.id === itemId);

        const saveBtn = document.createElement('div');
        saveBtn.classList.add('btn');
        saveBtn.innerText = this.options.strings.save;
        saveBtn.style.float = 'right';

        const wrapperDiv = document.createElement('div');

        const colorInput = document.createElement('input');
        colorInput.type = 'color';
        colorInput.id = 'selectColor';

        if (foundItem?.activeColor) {
            colorInput.value = foundItem.activeColor;
        }

        const pColorInput = document.createElement('input');
        pColorInput.type = 'color';
        pColorInput.id = 'selectPColor';

        if (foundItem?.passiveColor) {
            pColorInput.value = foundItem.passiveColor;
        }

        const lColorInput = document.createElement('input');
        lColorInput.type = 'color';
        lColorInput.id = 'selectPColor';

        if (foundItem?.logicColor) {
            lColorInput.value = foundItem.logicColor;
        }

        const idInput = document.createElement('input');
        idInput.type = 'text';
        idInput.id = 'typeId';
        idInput.value = item.id;
        idInput.className = 'input';

        const colorLabel = document.createElement('label');
        colorLabel.innerText = this.options.strings.selectActiveColor;
        colorLabel.htmlFor = colorInput.id;
        colorLabel.className = 'label';

        const pColorLabel = document.createElement('label');
        pColorLabel.innerText = this.options.strings.selectPassiveColor;
        pColorLabel.htmlFor = pColorInput.id;
        pColorLabel.className = 'label';

        const lColorLabel = document.createElement('label');
        lColorLabel.innerText = this.options.strings.selectLogicColor;
        lColorLabel.htmlFor = lColorInput.id;
        lColorLabel.className = 'label';

        const br = document.createElement('br');

        const idLabel = document.createElement('label');
        idLabel.innerText = this.options.strings.selectId;
        idLabel.htmlFor = idInput.id;
        idLabel.className = 'label';

        const editorDiv = document.createElement('div');
        editorDiv.id = 'editor';
        editorDiv.style.height = '190px';

        const toolbarDiv = document.createElement('div');
        toolbarDiv.id = 'toolbar'

        wrapperDiv.appendChild(colorLabel);
        wrapperDiv.appendChild(colorInput);

        wrapperDiv.appendChild(br);

        wrapperDiv.appendChild(pColorLabel);
        wrapperDiv.appendChild(pColorInput);

        wrapperDiv.appendChild(br);

        wrapperDiv.appendChild(lColorLabel);
        wrapperDiv.appendChild(lColorInput);

        wrapperDiv.appendChild(br);

        wrapperDiv.appendChild(idLabel);
        wrapperDiv.appendChild(idInput);

        wrapperDiv.appendChild(editorDiv);
        wrapperDiv.appendChild(toolbarDiv);
        wrapperDiv.appendChild(saveBtn);

        this.modal.show(wrapperDiv, `${this.options.strings.edit} ${this.options.strings.item}`);

        saveBtn.addEventListener('click', e => {
            e.preventDefault();
            foundItem.activeColor = colorInput.value
            foundItem.passiveColor = pColorInput.value
            foundItem.logicColor = lColorInput.value
            foundItem.id = idInput.value
            this.toggleActiveState(item);
            this.modal.hide();
        })

    },
    generateUID: function () {
        return Date.now().toString(36) + Math.random().toString(36).substring(2);
    },
    isHTMLObject: function (el) {
        return el instanceof Element || el instanceof HTMLDocument
    },
    findClosestEmptyCell: function (startX = 1, startY = 1) {
        const cells = Array.from(this.grid.children);
        const emptyCells = cells
            .map((cell, index) => {
                if (!cell.hasChildNodes()) {
                    const x = index % this.xCells;
                    const y = Math.floor(index / this.xCells);
                    const distance = Math.sqrt(Math.pow(x - (startX - 1), 2) + Math.pow(y - (startY - 1), 2));
                    return { cell, x, y, distance };
                }
                return null;
            })
            .filter(cellInfo => cellInfo !== null);

        emptyCells.sort((a, b) => a.distance - b.distance);

        return emptyCells[0] || null;
    },
    showTextModal: function (item) {
        this.modal.hide();
        const saveBtn = document.createElement('div');
        saveBtn.classList.add('btn');
        saveBtn.innerText = this.options.strings.save;
        saveBtn.style.float = 'right';

        const itemId = item && item.id ? item.id.replace('cloned-', '').split('-')[0] : null;
        const foundItem = this.items.find(i => i.id === itemId);

        const wrapperDiv = document.createElement('div');

        const editorDiv = document.createElement('div');
        editorDiv.id = 'editor';
        editorDiv.style.height = '190px';

        const toolbarDiv = document.createElement('div');
        toolbarDiv.id = 'toolbar'

        wrapperDiv.appendChild(editorDiv);
        wrapperDiv.appendChild(toolbarDiv);
        wrapperDiv.appendChild(saveBtn);

        this.modal.show(wrapperDiv, item.innerHTML ? this.options.strings.edit : this.options.strings.add + ' ' + this.options.strings.text)

        const toolbarOptions = [
            ['bold', 'italic', 'underline'],
            [{'size': ['small', false, 'large', 'huge']}],
            [{'color': []}],
        ];

        const opts = {
            theme: "snow",
            modules: {
                toolbar: toolbarOptions
            }
        };

        const quill = new Quill("#editor", opts);

        if (item?.innerHTML) {
            quill.setContents(quill.clipboard.convert({html: item.innerHTML}))
        }

        saveBtn.addEventListener('click', e => {
            e.preventDefault();

            const textData = quill.getContents();

            if (foundItem) {
                foundItem.textData = textData.ops;
                item.innerHTML = quill.getSemanticHTML();
            } else {
                const newItem = {
                    id: this.generateUID(),
                    imageName: 'text',
                    rotation: 0,
                    mirror: false,
                    active: false,
                    textData: textData.ops
                };

                const closestCell = this.findClosestEmptyCell();
                if (closestCell) {
                    const { x, y } = closestCell;
                    newItem.x = x;
                    newItem.y = y;

                    const textNode = this.createItemElement({ item: newItem });
                    this.items.push(newItem);
                    this.placeItem(textNode, x, y);
                } else {
                    alert('No empty cell available to place the text.');
                }
            }
            this.saveSchemeToFile();
            this.modal.hide();
        })
    },
    showSchemeOptions: function () {
        this.modal.hide();
        const saveBtn = document.createElement('div');
        saveBtn.classList.add('btn');
        saveBtn.innerText = this.options.strings.save;
        saveBtn.style.float = 'right';

        const wrapperDiv = document.createElement('div');

        const XCellsInput = document.createElement('input');
        XCellsInput.type = 'number';
        XCellsInput.id = 'selectXCells';
        XCellsInput.className = 'input';
        XCellsInput.style.width = '50px';
        XCellsInput.style.marginRight = '15px';
        XCellsInput.value = this.xCells;

        const XCellsLabel = document.createElement('label');
        XCellsLabel.innerText = 'X cells ';
        XCellsLabel.htmlFor = XCellsInput.id;
        XCellsLabel.className = 'label';

        const YCellsInput = document.createElement('input');
        YCellsInput.type = 'number';
        YCellsInput.id = 'selectYCells';
        YCellsInput.style.width = '50px';
        YCellsInput.className = 'input';
        YCellsInput.value = this.yCells;

        const YCellsLabel = document.createElement('label');
        YCellsLabel.innerText = 'Y cells ';
        YCellsLabel.htmlFor = YCellsInput.id;
        YCellsLabel.className = 'label';

        const title = document.createElement('span');
        title.innerText = 'Grid size: ';

        wrapperDiv.appendChild(title);

        wrapperDiv.appendChild(XCellsLabel);
        wrapperDiv.appendChild(XCellsInput);

        wrapperDiv.appendChild(YCellsLabel);
        wrapperDiv.appendChild(YCellsInput);

        wrapperDiv.appendChild(saveBtn);

        this.modal.show(wrapperDiv, 'Scheme options')

        saveBtn.addEventListener('click', e => {
            e.preventDefault();
            this.deleteScheme()
            this.xCells = XCellsInput.value;
            this.yCells = YCellsInput.value;
            this.createScheme(XCellsInput.value, YCellsInput.value);
            if (this.items) {
                this.applySchemeItems(this.items);
            }
            this.saveSchemeToFile();
            this.modal.hide();
        })
    },
    showPredefinedGridModal: function () {
        this.modal.hide();
        const innerHtml = this.fromHTML('<div id="predefined-grid-list"></div>');
        predefinedGrids.forEach((grid, index) => {
            const gridItem = document.createElement('div');
            gridItem.innerText = grid.name;
            gridItem.classList.add('item');

            const applyBtn = document.createElement('button');
            const previewBtn = document.createElement('button');
            previewBtn.innerText = this.options.strings.view;
            applyBtn.innerText = this.options.strings.apply;
            previewBtn.classList.add('btn');
            applyBtn.classList.add('btn');

            gridItem.appendChild(applyBtn)
            gridItem.appendChild(previewBtn)
            applyBtn.addEventListener('click', () => this.applyPredefinedGrid(index));
            previewBtn.addEventListener('click', () => this.showGridPreview(grid));
            innerHtml.appendChild(gridItem);
        });
        this.modal.show(innerHtml, this.options.strings.selectPredefinedGrid)
    },

    showGridPreview: function (grid) {
        this.modal.hide();
        this.modal.show(this.createGridPreview(grid), grid.name)
    },
    createGridPreview: function (grid) {
        const previewGrid = document.createElement('div');
        previewGrid.classList.add('preview-grid');
        previewGrid.style.gridTemplateColumns = `repeat(${this.xCells}, 5%)`;
        previewGrid.style.gridTemplateRows = `repeat(${this.yCells}, 20%)`;

        for (let i = 0; i < this.xCells * this.yCells; i++) {
            const cell = document.createElement('div');
            cell.classList.add('preview-cell');
            previewGrid.appendChild(cell);
        }

        grid.items.forEach((item) => {
            const itemDiv = document.createElement('div');
            itemDiv.classList.add('item');
            if (images[item.imageName]) {
                itemDiv.style.backgroundImage = images[item.imageName];
            }
            itemDiv.style.transform = `rotate(${item.rotation || 0}deg) ${item.mirror ? 'scaleX(-1)' : ''}`;
            const cellIndex = item.y * this.xCells + item.x;
            previewGrid.children[cellIndex].appendChild(itemDiv);
        });

        return previewGrid;
    },
    applyPredefinedGrid: function (index) {
        this.modal.hide();
        const grid = predefinedGrids[index];
        this.applySchemeItems(grid.items);
    },

    changeActiveEl(item) {
        const imageName = item.getAttribute("data-image-name");
        const isActive = item.classList.contains('active');
        // const state = item.getAttribute("data-state");

        if (isActive) {
            item.style.backgroundImage = images[imageName + '_op'];
        } else {
            item.style.backgroundImage = images[imageName + '_cl'];
        }
    },

    addEventListenerForElementsList: function () {
        let toggler = document.getElementsByClassName("caret");
        let i;
        for (i = 0; i < toggler.length; i++) {
            toggler[i].addEventListener("click", function () {
                this.parentElement.querySelector(".nested").classList.toggle("active-list");
                this.classList.toggle("caret-down");
            });
        }
    }
};

document.addEventListener('DOMContentLoaded', () => {
    setTimeout(function () {
        dragDropUI.init(20, 10);
    }, 100);
});